"IAL"

  
************************************************************************
* 
*  INVERSE ASSEMBLER FOR THE 68010 MICROPROCESSOR 
* 
* 
************************************************************************
*
*  THE LOGIC ANALYZER CAPTURES 24 ADDRESS LINES, 16 DATA LINES AND 
*  8 STATUS LINES ON THE RISING EDGE OF LAS.
* 
*  THE 8 STATUS LINES FOR THIS INVERSE ASSEMBLER ARE: 
*
*           BIT 0  ---  R/LW  (CPU PIN 9)
*           BIT 1  ---  LLDS  (CPU PIN 8)
*           BIT 2  ---  LUDS  (CPU PIN 7)
*           BIT 3  ---  LVMA  (CPU PIN 19) 
*           BIT 4  ---  FC0  (CPU PIN 28)
*           BIT 5  ---  FC1  (CPU PIN 27)
*           BIT 6  ---  FC2  (CPU PIN 26)
*           BIT 7  ---  LBGACK  (CPU PIN 12) 
* 
************************************************************************


EA_TYPE         VARIABLE        0   ; TYPE OF EA REQUESTED
REG_FIELD       VARIABLE        0   ; VALUE OF THE REGISTER FIELD 
MODE_FIELD      VARIABLE        0   ; VALUE OF THE MODE FIELD 
SIZE_FIELD      VARIABLE        0   ; VALUE OF THE SIZE FIELD 

DISASSEMBLY_BLOCK  VARIABLE     0   ; TRUE/FALSE - IS BLOCK ENCOUNTERED?
  
RD_STATUS       VARIABLE        0   ; HIGH/LOW BYTE READ FAILURES 
HIGH_BYTE       VARIABLE        0   ; HIGH ORDER 32 BIT ADDRESS BYTE
LOW_BYTE        VARIABLE        0   ; LOW ORDER 32 BIT ADDRESS BYTE 
INT             VARIABLE        0   ; TRUE = INTERRUPT ACK
SR_ACCESS       VARIABLE        0   ; TRUE = SR ACCESS OK, FALSE OTHERWISE
  
TEMP1           VARIABLE        0   ; GENERAL PURPOSE TEMPORARY 
  
REG_MASK        VARIABLE        0   ; MOVEM REGISTER MASK 
  
PREFETCH_LOC    VARIABLE        0   ; TELLS WHERE PREFETCH IS EXPECTED
  
REQUESTED_ADDRESS  VARIABLE     0   ; ADDRESS WE ARE LOOKING FOR
  
TEMP_DATA       VARIABLE        0   ; STORES UPPER BYTE OF 68008 INSTRUCTION
  
ABSOLUTE        VARIABLE        0   ; VARIABLE THAT TELLS IF THE ADDRESS IS 
                                    ;  OFFSET OR ABSOLUTE DUE TO OVERFLOW
INT_VAL        VARIABLE    0FFFFFH  ; VARIABLE FOR INT VS CPU SPACE OPEREATIONS
  
************************************************************************
  
ML2             CONSTANT        03H             2 LSB MASK
ML3             CONSTANT        07H             3 LSB MASK
ML4             CONSTANT        0FH             4 LSB MASK
ML8             CONSTANT        0FFH            8 LSB MASK
ML20            CONSTANT        00FFFFFH        20 LSB MASK 
ML24            CONSTANT        0FFFFFFH        24 LSB MASK 
  
INSTR_LINE      CONSTANT        1               FLAG FOR INSTR LINE 
NOT_INSTR_LINE  CONSTANT        0               FLAG FOR NON-INSTR LINE 
  
UNUSED_PREFETCH CONSTANT        1               TAG FOR PREFETCHED INSTR
OPERAND_USED    CONSTANT        2               TAG FOR USED OPERANDS 
  
FALSE           CONSTANT        0 
TRUE            CONSTANT        1 
  
BYTE            CONSTANT        00B             BYTE TRANSFER INDICATOR 
WORD            CONSTANT        01B             WORD TRANSFER INDICATOR 
LONG            CONSTANT        10B             LONG TRANSFER INDICATOR 
  
TAG_COLUMN      CONSTANT        9               COLUMN FOR TAG DISPLAY
  
STATUS_MASK     CONSTANT        10111111B       MASK 68010
STATUS_VALUE    CONSTANT        10101001B       VALUE 68010 
  
SUPR_DATA       ASCII           "supr data" 
SUPR_PROG       ASCII           "supr program"
USER_DATA       ASCII           "user data" 
USER_PROG       ASCII           "user program"
READ            ASCII           "read"
WRITE           ASCII           "write" 
TYPE_6800       ASCII           " (6800)" 
DMA             ASCII           "DMA" 
FOUR_STAR       ASCII           "****"
COMMA           ASCII           "," 
D               ASCII           "D" 
A               ASCII           "A" 
B               ASCII           "B" 
W               ASCII           "W" 
L               ASCII           "L" 
NUM_SIGN        ASCII           "#" 
OR_OP           ASCII           "OR"
AND_OP          ASCII           "AND" 
SUB_OP          ASCII           "SUB" 
EOR_OP          ASCII           "EOR" 
ADD_OP          ASCII           "ADD" 
CLR_OP          ASCII           "CLR" 
MOVE_OP         ASCII           "MOVE"
SR              ASCII           "SR"
CCR             ASCII           "CCR" 
USP             ASCII           "USP" 
TWO_STAR        ASCII           "**"
S               ASCII           "S" 
RIGHT_BRACKET   ASCII           "]" 
R               ASCII           "R" 
PERIOD          ASCII           "." 
ILLEGAL         ASCII           "Illegal" 
  
                SEARCH_LIMIT    34
  
************************************************************************
BIN3_FMT        FORMAT          3,BIN,3         3 BITS, BIN, 3 DIGITS 
DEC1_FMT_3      FORMAT          3,DEC,1         3 BITS, DEC, 1 DIGIT
HEX1_FMT        FORMAT          4,HEX,1         4 BITS, HEX, 1 DIGIT
HEX1_FMT_3      FORMAT          3,HEX,1         3 BITS, HEX, 1 DIGIT
HEX2_FMT        FORMAT          8,HEX,2         8 BITS, HEX, 2 DIGITS 
HEX2_FMT_5      FORMAT          5,HEX,2         5 BITS, HEX, 2 DIGITS 
HEX4_FMT        FORMAT         16,HEX,4        16 BITS, HEX, 4 DIGITS
HEX6_FMT        FORMAT         24,HEX,6        24 BITS, HEX, 6 DIGITS
HEX8_FMT        FORMAT         32,HEX,8        32 BITS, HEX, 8 DIGITS 
  
************************************************************************
  
   LABEL_TITLE ^68010 Mnemonic^
   BASE_TITLE  ^      hex^
   DEFAULT_WIDTH 31 
  
   LOAD ML20
   STORE INT_VAL
  
INITIALIZE
    SET RETURN_FLAGS,INSTR_LINE  PRESET TO INSTRUCTION LINE
    SET SR_ACCESS,FALSE          INITIAL CONDITION, DONT ALLOW SR ACCESS 
    SET DISASSEMBLY_BLOCK,FALSE  INITIALLY NO INVERSE ASSEMBLY BLOCK
   
    LOAD INPUT_ADDRESS           GET ADDRESS OF FIRST STATE PASSED
    STORE REQUESTED_ADDRESS      SAVE THIS AS THE ADDRESS WE'RE LOOKING AT
  
    IF INPUT_ERROR <> 0 THEN GOTO DATA_ERROR
  
    LOAD  STATUS_MASK       ; GET PROGRAM READ STATUS MASK
    STORE QUALIFY_MASK           ... AND SAVE AS INPUT QUALIFIER MASK 
    LOAD  STATUS_VALUE      ; GET PROGRAM READ STATUS VALUE
    STORE QUALIFY_VALUE          ... AND SAVE AS INPUT QUALIFIER VALUE
  
CHECK_INPUT_TAG 
    LOAD INPUT_TAG                            GET THE CURRENT TAG VALUE
    IF 17,16 <> 0 THEN GOTO OPCODE_DECODE     OK TO DISASSEMBLE 

*     ELSE ONLY DISPLAY STATUS INFORMATION
    
    LOAD INPUT_STATUS 
    AND QUALIFY_MASK
    IF 7,0 = QUALIFY_VALUE THEN GOTO FLAGGED_AS_INSTR 
  
************************************************************************
  
DISPLAY_STATUS
  
       SET RETURN_FLAGS,NOT_INSTR_LINE          DONT FLAG THIS LINE 

FLAGGED_AS_INSTR
       POSITION ABS,3 
       LOAD INPUT_DATA
       OUTPUT ACCUMULATOR,HEX4_FMT              DISPLAY THE INPUT DATA 
  
       POSITION REL,1 
       LOAD INPUT_TAG 
       IF 1,0 <> UNUSED_PREFETCH THEN GOTO NOT_PREFETCH 
          OUTPUT "unused prefetch"
          RETURN
  
NOT_PREFETCH
       LOAD INPUT_STATUS
       IF 7,7 = 1 THEN GOTO NOT_DMA
          OUTPUT DMA
          RETURN

NOT_DMA
       CASE_OF 6,4
          CALL   UNKNOWN
          OUTPUT USER_DATA
          OUTPUT USER_PROG
          CALL   UNKNOWN
          CALL   UNKNOWN
          OUTPUT SUPR_DATA
          OUTPUT SUPR_PROG
          CALL   INT_ACK
       CASE_END 
       IF INT = TRUE THEN GOTO NO_R_W_DISP       INTERRUPT ACK IS ALWAYS READ 
       POSITION REL,1 
       CASE_OF 0,0
          OUTPUT WRITE
          OUTPUT READ 
       CASE_END 

NO_R_W_DISP
       CASE_OF 3,3
          OUTPUT TYPE_6800      INDICATE A 6800 CYCLE 
          NOP                   NOTHING FOR NORMAL 68000 CYCLE
       CASE_END 
  
*  IF THE TRANSFER IS A BYTE TRANSFER THE UNDEFINED BYTE IS xxed OUT. 
*  UPPER AND LOWER DATA STROBE TELL WHICH BYTE OR BYTES ARE VALID.
*  UPPER DATA STROBE IS INPUT ON STATUS BIT 2 AND LOWER DATA
*  STROBE IS INPUT ON STATUS BIT 1. 
  
       CASE_OF 2,1
          RETURN                WORD XFER, NO DONT CARES
          POSITION ABS,5        POSITION TO xx THE LOWER BYTE 
          POSITION ABS,3        POSITION TO xx THE UPPER BYTE 
          RETURN                UNDEFINED TRANSFER
       CASE_END 
       OUTPUT "xx"              OUTPUT THE DON'T CARES
     
       RETURN    		EXIT INVERSE ASSEMBLER
  
  
************************************************************************
  
OPCODE_DECODE 
  
   LOAD INPUT_TAG               GET THE TAG FOR THIS STATE
   IF 1,0 = OPERAND_USED THEN GOTO DISPLAY_STATUS 
   IF 1,0 = UNUSED_PREFETCH THEN GOTO DISPLAY_STATUS
  
   LOAD INPUT_STATUS            GET THE STATUS OF THE INPUT 
   AND  QUALIFY_MASK            GET ONLY THE DESIRED BITS FOR PROGRAM READ
   IF 7,0 <> QUALIFY_VALUE THEN GOTO DISPLAY_STATUS     BRIF NOT OPCODE 
  
   LOAD INPUT_DATA              GET THE DATA THAT WAS READ

MEM_OPC_DEC 
   LOAD INPUT_DATA              GET THE DATA THAT WAS READ
   CASE_OF 15,12
      GOTO Bit_MOVEP_Imm
      GOTO MOVE_B_L_W 
      GOTO MOVE_B_L_W 
      GOTO MOVE_B_L_W 
      GOTO Misc 
      GOTO ADDQ_SUBQ_Scc_DBcc 
      GOTO Bcc
      GOTO MOVEQ
      GOTO OR_DIV_BCD_AND_MUL_EXG 
      GOTO Math_Extended
      GOTO UNIMPLEMENTED
      GOTO CMP_EOR
      GOTO OR_DIV_BCD_AND_MUL_EXG 
      GOTO Math_Extended
      GOTO Shift_Rotate 
      GOTO UNIMPLEMENTED
   CASE_END 
  
************************************************************************
  
Bit_MOVEP_Imm 
   IF 8,8 = 1 THEN GOTO Dynamic_MOVEP 
   CASE_OF 11,9 
      OUTPUT OR_OP               0000 0000 XXXX XXXX
      OUTPUT AND_OP              0000 0010 XXXX XXXX
      OUTPUT SUB_OP              0000 0100 XXXX XXXX
      OUTPUT ADD_OP              0000 0110 XXXX XXXX
      GOTO Static                0000 1000 XXXX XXXX
      OUTPUT EOR_OP              0000 1010 XXXX XXXX
      OUTPUT "CMP"               0000 1100 XXXX XXXX
      GOTO MOVES                0000 1110 XXXX XXXX 
   CASE_END 
   OUTPUT "I"                   ALL OF THE ABOVE ARE IMMEDIATE
   CALL SHOW_SIZE               DISPLAY THE SIZE INDICATOR
   CALL IMMEDIATE               DISPLAY THE IMMEDIATE DATA
   OUTPUT COMMA                   SEPARATE OPERAND FIELDS 
  
*  NOW SET THE VARIABLE THAT ALLOWS/DISALLOWS EFFECTIVE ADDRESS ACCESS
*  TO THE STATUS REGISTER 
  
   LOAD INITIAL_DATA            GET THE ORIGINAL OPCODE BACK
   CASE_OF 11,9 
      SET SR_ACCESS,TRUE        ORI CAN ACCESS STATUS REGISTER
      SET SR_ACCESS,TRUE        ANDI CAN ACCESS STATUS REGISTER 
      SET SR_ACCESS,FALSE       SUBI CANNOT ACCESS STATUS REGISTER
      SET SR_ACCESS,FALSE       ADDI CANNOT ACCESS STATUS REGISTER
      NOP                       CANT GET HERE (BIT STATIC INSTR)
      SET SR_ACCESS,TRUE        EORI CAN ACCESS STATUS REGISTER 
      SET SR_ACCESS,FALSE       CMPI CANNOT ACCESS STATUS REGISTER
      NOP                       CANT GET HERE (ILLEGAL OPCODE)
   CASE_END 
   CALL EA_DISP                 DISPLAY THE EFFECTIVE ADDRESS 
   IF SR_ACCESS = FALSE THEN RETURN             CANNOT ACCESS SR/CCR
   LOAD INITIAL_DATA            GET THE ORIGINAL OPCODE 
   IF 5,0 <> 111100B THEN RETURN    COULD, BUT DIDNT ACCESS SR/CCR
   RETURN 
************************************************************************
  
MOVES 
    OUTPUT MOVE_OP
    OUTPUT S
    CALL SHOW_SIZE
    CALL READ_NEXT_OPERAND
    IF INPUT_ERROR <>0 THEN GOTO INCOMPLETE_OPCODE
    LOAD INPUT_DATA 
    STORE TEMP_DATA 
    CASE_OF 11,11 
            CALL EA_DISP
            CALL REG_DISP 
    CASE_END
    OUTPUT COMMA
    LOAD TEMP_DATA
    CASE_OF 11,11 
            CALL REG_DISP 
            CALL EA_DISP
    CASE_END
    RETURN
  
REG_DISP
    LOAD TEMP_DATA
    CASE_OF 15,15 
            OUTPUT D
            OUTPUT A
    CASE_END
    AND 07000H
    ROTATE RIGHT,12 
    OUTPUT ACCUMULATOR,DEC1_FMT_3 
    RETURN
  
************************************************************************
  
Dynamic_MOVEP 
   IF 5,3 = 001B THEN GOTO MOVEP
   CALL BIT_MNEMONIC            DISPLAY THE BIT MANIPULATION INSTR
   POSITION ABS,TAG_COLUMN      MOVE OVER TO OPERAND FIELD
   CALL DATA_REG_11_9           DISPLAY THE REG NUMBER (BITS 11 - 9)
   OUTPUT COMMA 
   GOTO EA_DISP                 DISPLAY THE EFFECTIVE ADDRESS 
  
************************************************************************
  
BIT_MNEMONIC
  
*  DISPLAY THE BIT MANIPULATION INSTRUCTION (DYNAMIC/STATIC)
  
   OUTPUT B 
   CASE_OF 7,6
      OUTPUT "TST"
      OUTPUT "CHG"
      OUTPUT CLR_OP 
      OUTPUT "SET"
   CASE_END 
   OUTPUT PERIOD
   IF 5,3 = 000B THEN GOTO LONG_BIT             BRIF DYNAMIC BIT
      OUTPUT B
      SET SIZE_FIELD,BYTE       INDICATE BYTE SIZE TRANSFER 
      RETURN
LONG_BIT
      OUTPUT L
      SET SIZE_FIELD,LONG       INDICATE LONG SIZE TRANSFER 
      RETURN
  
************************************************************************
  
MOVEP 
   OUTPUT MOVE_OP 
   OUTPUT "P."              DISPLAY THE MNEMONIC
   CASE_OF 6,6
      OUTPUT W                SIZE = WORD 
      OUTPUT L                SIZE = LONG 
   CASE_END 
   CASE_OF 6,6
      SET SIZE_FIELD,WORD       SIZE = WORD 
      SET SIZE_FIELD,LONG       SIZE = LONG 
   CASE_END 
   SET MODE_FIELD,101B          SET TO ADDR INDIRECT + OFFSET MODE
   POSITION ABS,TAG_COLUMN      MOVE OVER TO THE OPERAND FIELD
   CASE_OF 7,7
      CALL EA_DISP_MODE_SET     DISPLAY THE MEMORY SOURCE LOCATION
      CALL DATA_REG_11_9        DISPLAY THE DATA REG SOURCE 
   CASE_END 
   LOAD INITIAL_DATA            GET BACK THE ORIGINAL OPCODE
   OUTPUT COMMA 
   CASE_OF 7,7
      CALL DATA_REG_11_9        DISPLAY THE DATA REG DESTINATION
      CALL EA_DISP_MODE_SET     DISPLAY THE MEMORY DESTINATION
   CASE_END 
   RETURN 
  
************************************************************************
  
Static
   CALL BIT_MNEMONIC            DISPLAY THE BIT MANIPULATION INSTR
   POSITION ABS,TAG_COLUMN      MOVE OVER TO THE OPERAND FIELD
   OUTPUT NUM_SIGN                   INDICATE IMMEDIATE DATA
   CALL READ_NEXT_OPERAND       GET THE BIT NUMBER OPERAND
   IF INPUT_ERROR = 0 THEN GOTO Static_OK 
      LOAD SIZE_FIELD           GET THE SIZE INDICATOR
      CASE_OF 1,1 
         OUTPUT "*"             0X = BYTE ie MODULO 8 BIT NUMBER
         OUTPUT TWO_STAR        1X = LONG ie MODULO 32 BIT NUMBER 
      CASE_END
      GOTO SNODEA               Static Not Ok Display Effective Address 
Static_OK 
   LOAD INPUT_DATA              GET THE BIT NUMBER SPECIFIER
   CASE_OF SIZE_FIELD 
      OUTPUT  ACCUMULATOR,HEX1_FMT_3            DISPLAY THE 3 BIT VALUE 
      NOP                       SHOULDN'T EVER GET HERE 
      OUTPUT ACCUMULATOR,HEX2_FMT_5             DISPLAY THE 5 BIT VALUE 
      NOP 
   CASE_END 
SNODEA
   OUTPUT COMMA 
   GOTO EA_DISP                 DISPLAY THE EFFECTIVE ADDRESS 
  
************************************************************************
  
MOVE_B_L_W
   OUTPUT MOVE_OP 
   IF 8,6 = 001B THEN OUTPUT A
   OUTPUT PERIOD
  
*  THE CONVENTIONAL SIZE ROUTINE (SHOW_SIZE) CANNOT BE USED BECAUSE 
*  THE MOVE INSTRUCTION USES A DIFFERENT SIZE ENCODING
  
   CASE_OF 13,12                SIZE FIELD
      NOP                       SHOULDNT EVER GET HERE
      OUTPUT B                BYTE TRANSFER 
      OUTPUT L                LONG TRANSFER 
      OUTPUT W                WORD TRANSFER 
   CASE_END 
   CASE_OF 13,12                SIZE FIELD
      NOP                       SHOULDNT EVER GET HERE
      SET SIZE_FIELD,BYTE       BYTE TRANSFER 
      SET SIZE_FIELD,LONG       LONG TRANSFER 
      SET SIZE_FIELD,WORD       WORD TRANSFER 
   CASE_END 
  
   POSITION ABS,TAG_COLUMN      POSITION AT THE TAG DISPLAY POSITION
   CALL SOURCE_EA               DISPLAY THE DESTINATION EFFECTIVE ADDR
   OUTPUT COMMA 
   LOAD INITIAL_DATA            GET THE OPCODE BACK AGAIN 
   GOTO DESTINATION_EA          DISPLAY THE SOURCE EFFECTIVE ADDRESS
  
************************************************************************
  
Misc
   IF 8,7 = 10B THEN GOTO ILLEGAL_OPCODE
   IF 8,8 = 1 THEN GOTO CHK_LEA 
   IF 11,11 = 0 THEN GOTO NEGX_CLR_NOT_MOVE 
   CASE_OF 10,9 
      GOTO NBCD_AND_MIX         0100 1000 XXXX XXXX 
      GOTO TST_TAS              0100 1010 XXXX XXXX 
      GOTO MOVEM_TO_REG         0100 1100 XXXX XXXX 
      GOTO MISC_JSR_JMP         0100 1110 XXXX XXXX 
   CASE_END 
   RETURN                       THIS RETURN CANT HAPPEN 
  
NBCD_AND_MIX
   CASE_OF 7,6
      OUTPUT "NBCD.B"           0100 1000 00XX XXXX 
      GOTO PEA_SWAP             0100 1000 01XX XXXX 
      GOTO MOVEM_EXT            0100 1000 10XX XXXX 
      GOTO MOVEM_EXT            0100 1000 11XX XXXX 
   CASE_END 
   POSITION ABS,TAG_COLUMN      MOVE OVER TO TAG FIELD
   GOTO EA_DISP                 DISPLAY THE EFFECTIVE ADDRESS 
  
PEA_SWAP
   SET TEMP1,0                  PRESET TO SWAP INSTRUCTION
   IF 5,3 <> 000B THEN SET TEMP1,1        SET TO PEA INSTRUCTION
   CASE_OF TEMP1
      OUTPUT "SWAP.W" 
      OUTPUT "PEA.L"
   CASE_END 
   POSITION ABS,TAG_COLUMN
   CASE_OF TEMP1
      CALL DATA_REG_2_0         SWAP INSTRUCTION
      CALL EA_DISP              PEA INSTRUCTION 
   CASE_END 
   RETURN 
  
CHK_LEA 
   CASE_OF 6,6
      OUTPUT "CHK.W"            0100 XXX1 10XX XXXX 
      OUTPUT "LEA.L"            0100 XXX1 11XX XXXX 
   CASE_END 
   POSITION ABS,TAG_COLUMN      MOVE OVER TO OPERAND FIELD
   CALL EA_DISP                 DISPLAY THE EFFECTIVE ADDRESS 
   LOAD INITIAL_DATA            RECOVER THE ORIGINAL OPCODE 
   OUTPUT COMMA 
   CASE_OF 6,6
      CALL DATA_REG_11_9        BOUND IS IN DATA REGISTER 
      CALL ADDR_REG_11_9        EFFECTIVE ADDRESS GOES TO ADDR REG
   CASE_END 
   RETURN 
  
NEGX_CLR_NOT_MOVE 
   IF 7,6 = 11B THEN GOTO MOVE_SR_CCR           0100 0XX0 11XX XXXX 
   CASE_OF 10,9 
      OUTPUT "NEGX" 
      OUTPUT CLR_OP 
      OUTPUT "NEG"
      OUTPUT "NOT"
   CASE_END 
   CALL SHOW_SIZE               DISPLAY THE OPERATION SIZE
   CALL EA_DISP                 DISPLAY THE EFFECTIVE ADDRESS 
   RETURN 
  
MOVE_SR_CCR 
   OUTPUT MOVE_OP 
   OUTPUT PERIOD
   CASE_OF 10,9 
      OUTPUT W
      OUTPUT W
      OUTPUT B
      OUTPUT W
   CASE_END 
   CASE_OF 10,9 
      SET SIZE_FIELD,WORD 
      SET SIZE_FIELD,WORD       CANT GET HERE FOR 68000/08
      SET SIZE_FIELD,BYTE 
      SET SIZE_FIELD,WORD 
   CASE_END 
   POSITION ABS,TAG_COLUMN
   CASE_OF 10,9 
      OUTPUT SR 
      OUTPUT CCR              ILLEGAL INSTRUCTION, CANT GET HERE FOR 68000/08 
      CALL EA_DISP              DISPLAY THE SOURCE ADDRESS
      CALL EA_DISP              DISPLAY THE SOURCE ADDRESS
   CASE_END 
   OUTPUT COMMA 
   LOAD INITIAL_DATA            GET THE ORIGINAL OPCODE BACK
   CASE_OF 10,9 
      CALL EA_DISP              DISPLAY THE DESTINATION ADDRESS 
      CALL EA_DISP              ILLEGAL INSTRUCTION, CANT GET HERE FOR 68000/08 
      OUTPUT CCR
      OUTPUT SR 
   CASE_END 
   LOAD INITIAL_DATA            GET THE ORIGINAL OPCODE BACK
   IF 10,9 < 2 THEN RETURN      NO UNUSED PREFETCH TO MARK
* 
*  NOW, LOOK AT THE EFFECTIVE ADDRESS MODE FIELD TO DETERMINE THE NUMBER
*  OF PROGRAM READ STATES REQUIRED FOR THIS INSTRUCTION. THIS VALUE PLUS
*  ONE WILL THEN POINT TO THE PROPER LOCATION FOR THE UNUSED PREFETCH 
*  STATE. 
* 
   CASE_OF 5,3
      SET PREFETCH_LOC,+1       DATA REGISTER DIRECT
      NOP                       ADDRESS REGISTER DIRECT    (**ILLEGAL**)
      SET PREFETCH_LOC,+1       ADDRESS REGISTER INDIRECT 
      SET PREFETCH_LOC,+1       ADDRESS REGISTER INDIRECT POST-INCREMENT
      SET PREFETCH_LOC,+1       ADDRESS REGISTER INDIRECT PRE-DECREMENT 
      SET PREFETCH_LOC,+2       ADDRESS REGISTER INDIRECT W/DISPLACEMENT
      SET PREFETCH_LOC,+2       ADDR REG IND W/DISPLACEMENT AND INDEX 
      CALL LOOK_AT_REG_FIELD    FOR MODE = 7, REG FIELD MUST BE EXAMINED
   CASE_END 
   GOTO MARK_PREFETCH           MARK THE PREFETCH 
  
LOOK_AT_REG_FIELD 
   CASE_OF 2,0                  LOOKING NOW AT THE REG FIELD
      SET PREFETCH_LOC,+2       ABSOLUTE ADDRESS (SHORT)
      SET PREFETCH_LOC,+3       ABSOLUTE ADDRESS (LONG) 
      SET PREFETCH_LOC,+2       PC RELATIVE W/DISPLACEMENT
      SET PREFETCH_LOC,+2       PC RELATIVE W/DISPLACEMENT AND INDEX
      SET PREFETCH_LOC,+2       IMMEDIATE DATA
   CASE_END 
   RETURN 
  
MOVEM_EXT 
   IF 5,3 = 000B THEN GOTO EXT
MOVEM 
   OUTPUT MOVE_OP 
   OUTPUT "M."
   CASE_OF 6,6
      OUTPUT W
      OUTPUT L
   CASE_END 
   CALL READ_NEXT_OPERAND       READ THE MASK WORD
   STORE REG_MASK               SAVE THE MASK WORD
   IF INPUT_ERROR <> 0 THEN SET REG_MASK,0
   POSITION ABS,TAG_COLUMN
   LOAD INITIAL_DATA            GET THE ORIGINAL OPCODE BACK
   CASE_OF 10,10
      CALL MOVEM_MASK 
      CALL EA_DISP
   CASE_END 
   OUTPUT COMMA 
   LOAD INITIAL_DATA            GET BACK THE ORIGINAL OPCODE
   CASE_OF 10,10
      CALL EA_DISP
      CALL MOVEM_MASK 
   CASE_END 
   RETURN 
  
MOVEM_MASK
   OUTPUT "rm=" 
   LOAD REG_MASK
   IF 15,0 <> 0 THEN GOTO MASK_FOUND
      OUTPUT FOUR_STAR          IF NO MASK WAS FOUND
      RETURN
MASK_FOUND
   OUTPUT ACCUMULATOR,HEX4_FMT
   RETURN 
  
  
  
EXT 
   OUTPUT "EXT."
   CASE_OF 6,6
      OUTPUT W
      OUTPUT L
   CASE_END 
   POSITION ABS,TAG_COLUMN
   GOTO DATA_REG_2_0
  
TST_TAS 
   IF 7,6 <> 11B THEN GOTO TST
      OUTPUT "TAS.B"
      GOTO MNEMONIC_DISPLAYED 
TST 
   OUTPUT "TST" 
   CALL SHOW_SIZE 
  
MNEMONIC_DISPLAYED
   POSITION ABS,TAG_COLUMN      MOVE OVER TO THE OPERAND FIELD
   GOTO EA_DISP                 DISPLAY THE EFFECTIVE ADDRESS 
  
MOVEM_TO_REG
   IF 7,7 = 0 THEN GOTO ILLEGAL_OPCODE
   GOTO MOVEM 
  
MISC_JSR_JMP
   CASE_OF 8,6
      GOTO ILLEGAL_OPCODE 
      GOTO TRAP_MIX 
      OUTPUT "JSR"
      OUTPUT "JMP"
   CASE_END 
   POSITION ABS,TAG_COLUMN
   GOTO EA_DISP 
*  CALL MARK_PREFETCH           DUMP THE UNUSED PREFETCH
  
TRAP_MIX
   CASE_OF 5,4
      GOTO TRAP 
      GOTO LINK_UNLK
      GOTO MOVE_USP 
      NOP 
   CASE_END 
   IF 5,3 = 111B THEN GOTO MOVEC
   CASE_OF 2,0
      OUTPUT "RESET"
      OUTPUT "NOP"
      OUTPUT "STOP" 
      OUTPUT "RTE"
      GOTO  RTD 
      OUTPUT "RTS"
      OUTPUT "TRAPV"
      OUTPUT "RTR"
   CASE_END 
   IF 2,0 <> 2 THEN GOTO NO_OPERAND 
      POSITION ABS,TAG_COLUMN 
      OUTPUT NUM_SIGN 
      CALL READ_NEXT_OPERAND
      IF INPUT_ERROR <>0 THEN GOTO WORD_ERROR 
STOP_OK 
   OUTPUT ACCUMULATOR,HEX4_FMT
   RETURN 
  
RTD 
    OUTPUT "RTD"
    POSITION ABS,TAG_COLUMN 
    OUTPUT NUM_SIGN 
    CALL READ_NEXT_OPERAND
    IF INPUT_ERROR <>0 THEN GOTO WORD_ERROR 
    LOAD INPUT_DATA 
    OUTPUT ACCUMULATOR,HEX4_FMT 
    RETURN
  
MOVEC 
    OUTPUT MOVE_OP
    OUTPUT "C.L"
    POSITION ABS,TAG_COLUMN 
    CALL READ_NEXT_OPERAND
    IF INPUT_ERROR <>0 THEN GOTO INCOMPLETE_OPCODE
    LOAD INPUT_DATA 
    STORE TEMP_DATA 
    LOAD INITIAL_DATA 
    CASE_OF 0,0 
            CALL DISP_CONTROL_REG 
            CALL REG_DISP 
    CASE_END
    OUTPUT COMMA
    LOAD INITIAL_DATA 
    CASE_OF 0,0 
            CALL REG_DISP 
            CALL DISP_CONTROL_REG 
    CASE_END
    RETURN
  
DISP_CONTROL_REG
    LOAD TEMP_DATA
    IF 11,0=0 THEN OUTPUT "SFC" 
    IF 11,0=1 THEN OUTPUT "DFC" 
    IF 11,0=800H THEN OUTPUT USP
    IF 11,0=801H THEN OUTPUT "VBR"
    IF 10,1<> 0 THEN GOTO ILLEGAL_OPERAND 
    RETURN
  
NO_OPERAND
   IF 2,0 < 3 THEN RETURN       NO PREFETCH TO BE MARKED
   IF 2,0 = 6 THEN RETURN       CANT MARK PREFETCH FOR TRAPV
   SET PREFETCH_LOC,+1
   GOTO MARK_PREFETCH 
  
TRAP
   OUTPUT "TRAP"
   POSITION ABS,TAG_COLUMN
   OUTPUT NUM_SIGN
   AND ML4                      VECTOR NUMBER IN LOWER 4 BITS ONLY
   OUTPUT ACCUMULATOR,HEX1_FMT
   SET PREFETCH_LOC,+1
   GOTO MARK_PREFETCH           DUMP UNUSED PREFETCH
  
LINK_UNLK 
   CASE_OF 3,3
      OUTPUT "LINK" 
      OUTPUT "UNLK" 
   CASE_END 
   POSITION ABS,TAG_COLUMN
   CALL ADDR_REG_2_0
   IF 3,3 = 1 THEN RETURN 
   OUTPUT ",#"
   CALL READ_NEXT_OPERAND 
   IF INPUT_ERROR <> 0 THEN GOTO WORD_ERROR 
LINK_UNLK_OK
   OUTPUT ACCUMULATOR,HEX4_FMT  DISPLAY THE 16 BIT DISPLACEMENT 
   RETURN 
  
MOVE_USP
   OUTPUT MOVE_OP 
   OUTPUT ".L"
   POSITION ABS,TAG_COLUMN
   CASE_OF 3,3
      CALL ADDR_REG_2_0 
      OUTPUT USP
   CASE_END 
   OUTPUT COMMA 
   CASE_OF 3,3
      OUTPUT USP
      CALL ADDR_REG_2_0 
   CASE_END 
   RETURN 
  
************************************************************************
  
ADDQ_SUBQ_Scc_DBcc
   IF 7,6 <> 11B THEN GOTO ADDQ_SUBQ            ADDQ & SUBQ INSTRUCTIONS
   IF 5,3 = 001B THEN GOTO DBcc 
      OUTPUT S
      GOTO Scc_DBcc 
DBcc
      OUTPUT "DB" 
  
Scc_DBcc
   CALL CONDITION_CODE          DISPLAY THE CONDITION CODE
   IF 5,3 = 001B THEN GOTO DBcc_2 
      OUTPUT ".B" 
      POSITION ABS,TAG_COLUMN      MOVE OVER TO THE TAG LOCATION
      GOTO EA_DISP
  
DBcc_2
   POSITION ABS,TAG_COLUMN      MOVE OVER TO THE TAG LOCATION 
   OUTPUT D                   A DATA REGISTER IS BEING USED 
   OUTPUT ACCUMULATOR,DEC1_FMT_3  DISPLAY THE REGISTER NUMBER 
   OUTPUT COMMA 
   CALL READ_NEXT_OPERAND 
   IF INPUT_ERROR <> 0 THEN GOTO WORD_ERROR 
DBcc_DISP_OK
   IF 15,15 = 1 THEN INCLUSIVE_OR 0FF0000H    SIGN EXTENSION
   ADD INPUT_ADDRESS            ADD PC OF DISPLACEMENT WORD TO DISPLACEMENT 
ADDRESS_OUTPUT
   AND ML24                     ONLY INTERESTED IN LOWER 24 BITS
   IF_NOT_MAPPED THEN OUTPUT ACCUMULATOR,HEX6_FMT 
   RETURN 
  
************************************************************************
  
ADDQ_SUBQ 
   CASE_OF 8,8
      OUTPUT "ADDQ" 
      OUTPUT "SUBQ" 
   CASE_END 
   CALL SHOW_SIZE               DISPLAY THE SIZE OF THE OPERATION 
   CALL IMM_DATA_11_9           DISPLAY THE 3 BIT IMMEDIATE DATA
   OUTPUT COMMA 
   GOTO EA_DISP                 DISPLAY THE EFFECTIVE ADDRESS 
  
************************************************************************
  
Bcc 
   OUTPUT B 
   CALL CONDITION_CODE          DISPLAY THE APPLICABLE CONDITION
  
   IF 7,0 <> 0 THEN GOTO EIGHT_BIT     BRIF AN 8 BIT DISPLACEMENT VALUE 
      OUTPUT ".W"               DISPLACEMENT WAS 16 BIT WORD
      SET SIZE_FIELD,WORD       OPERATION SIZE = WORD 
      CALL READ_NEXT_OPERAND           GET THE 16 BIT DISPLACEMENT VALUE
      IF 15,15 = 1 THEN INCLUSIVE_OR 0FF0000H      SIGN EXTENSION 
      IF INPUT_ERROR = 0 THEN GOTO DISP_OK
         OUTPUT "********"
         RETURN 
  
EIGHT_BIT 
   OUTPUT ".B"                  DISPLACEMENT WAS 8 BIT BYTE 
   SET SIZE_FIELD,BYTE          OPERATION SIZE = BYTE 
   AND ML8                      INTERESTED IN LOWER 8 BITS ONLY 
   IF 7,7 = 1 THEN INCLUSIVE_OR 0FFFF00H      EXTEND THE SIGN 
  
DISP_OK 
   POSITION ABS,TAG_COLUMN      MOVE OVER TO THE OPERAND FIELD
   ADD INPUT_ADDRESS            ADD ADDRESS OF DISPLACEMENT WORD
   AND ML24                     ONLY INTERESTED IN LOWER 24 BITS
   IF SIZE_FIELD = BYTE THEN ADD 2     PC POINTING AHEAD 2 LOCATIONS
   CALL ADDRESS_OUTPUT
   LOAD INITIAL_DATA            RELOAD THE INITIAL OPCODE 
   IF 15,9 <> 0110000B THEN RETURN     BRIF NOT BRA OR BSR
   IF 7,0 = 0 THEN RETURN       NO PREFETCH IF BSR.W OR BRA.W 
   SET PREFETCH_LOC,+1
   GOTO MARK_PREFETCH 
  
************************************************************************
  
MOVEQ 
   OUTPUT MOVE_OP 
   OUTPUT "Q.L" 
   POSITION ABS,TAG_COLUMN      MOVE OVER TO TAG COLUMN 
   AND ML8                      GET THE 8 BIT DATA FIELD
   IF 7,7 = 1 THEN INCLUSIVE_OR 0FFFFFF00H      SIGN EXTEND TO 32 BITS
   OUTPUT NUM_SIGN                   IMMEDIATE DATA INDICATOR 
   OUTPUT ACCUMULATOR,HEX8_FMT  DISPLAY THE 32 BIT IMMEDIATE VALUE
   OUTPUT COMMA 
   GOTO DATA_REG_11_9           DISPLAY THE DATA REGISTER IN 11 THRU 9
  
************************************************************************
  
Math_Extended 
   CASE_OF 14,14
      OUTPUT SUB_OP 
      OUTPUT ADD_OP 
   CASE_END 
   IF 7,6 = 11B THEN GOTO SUBA_ADDA 
   IF 8,8 <> 1 THEN GOTO NOT_EXTENDED 
      IF 5,4 = 00B THEN GOTO EXTENDED_MATH
  
NOT_EXTENDED
   CALL SHOW_SIZE               DISPLAY THE SIZE OF THE OPERATION 
   POSITION ABS,TAG_COLUMN      MOVE OVER TO THE OPERAND FIELD
   CASE_OF 8,8
      CALL EA_DISP              DISPLAY THE SOURCE EFFECTIVE ADDRESS
      CALL DATA_REG_11_9        DISPLAY THE SOURCE DATA REGISTER
   CASE_END 
   OUTPUT COMMA 
   LOAD INITIAL_DATA            GET BACK THE ORIGINAL OPCODE
   CASE_OF 8,8
      CALL DATA_REG_11_9        DISPLAY THE DESTINATION DATA REGISTER 
      CALL EA_DISP              DISPLAY THE DESTINATION EFFECTIVE ADDR
   CASE_END 
   RETURN 
  
SUBA_ADDA 
   OUTPUT "A."
A_ADDRESSING
   CASE_OF 8,8
      OUTPUT W
      OUTPUT L
   CASE_END 
   CASE_OF 8,8
      SET SIZE_FIELD,WORD       WORD OPERAND
      SET SIZE_FIELD,LONG       LONG OPERAND
   CASE_END 
   POSITION ABS,TAG_COLUMN      MOVE OVER TO THE OPERAND FIELD
   CALL EA_DISP                 DISPLAY THE SOURCE ADDRESS
   OUTPUT COMMA 
   LOAD INITIAL_DATA            GET THE ORIGINAL OPCODE 
   GOTO ADDR_REG_11_9           DISPLAY THE ADDRESS DESTINATION ADDR
  
************************************************************************
  
CMP_EOR 
   IF 7,6 = 11B THEN GOTO CMPA
   IF 8,8 = 1 THEN GOTO CMPM_EOR
      OUTPUT "CMP"
      CALL SHOW_SIZE            DISPLAY THE OPERATION SIZE
      CALL EA_DISP              DISPLAY THE EA SOURCE LOCATION
      OUTPUT COMMA
      GOTO DATA_REG_11_9        DISPLAY THE DATA REGISTER NAME
  
CMPA
   OUTPUT "CMPA." 
   GOTO A_ADDRESSING
  
CMPM_EOR
   IF 5,3 <> 001B THEN GOTO EOR 
      OUTPUT "CMPM" 
      CALL SHOW_SIZE            DISPLAY THE OPERATION SIZE
      POSITION ABS,TAG_COLUMN   MOVE OVER TO THE OPERAND DISPLAY FIELD
      OUTPUT "[xx]+,[xx]+"      DISPLAY OPERAND (xx WILL BE BACKFILLED) 
      POSITION REL,-10          MOVE BACK OVER TO SRC REG # FIELD 
      CALL ADDR_REG_2_0         DISPLAY THE 3 BIT REGISTER NUMBER 
      POSITION REL,+4           MOVE OVER TO THE DEST REG FIELD 
      CALL ADDR_REG_11_9        DISPLAY THE 3 BIT ADDRESS REGISTER NAME 
      POSITION REL,+2           MOVE TO THE END OF THE OPERAND
      RETURN
  
EOR 
   OUTPUT EOR_OP
   CALL SHOW_SIZE               DISPLAY THE OPERATION SIZE
   CALL DATA_REG_11_9           DISPLAY THE SOURCE DATA REGISTER
   OUTPUT COMMA 
   GOTO EA_DISP                 DISPLAY THE DESTINATION EFFECTIVE ADDR
  
************************************************************************
  
OR_DIV_BCD_AND_MUL_EXG
  
   IF 7,6 = 11B THEN GOTO DIV_MUL_U_S 
   IF 14,14 = 0 THEN GOTO NOT_EXG 
      IF 8,4 = 10100B THEN GOTO EXG 
      IF 8,4 = 11000B THEN GOTO EXG 
NOT_EXG 
   IF 8,4 = 10000B THEN GOTO SBCD_ABCD
   CASE_OF 14,14
      OUTPUT OR_OP
      OUTPUT AND_OP 
   CASE_END 
   GOTO NOT_EXTENDED
  
DIV_MUL_U_S 
   CASE_OF 14,14
      OUTPUT "DIV"
      OUTPUT "MUL"
   CASE_END 
   CASE_OF 8,8
      OUTPUT "U"                UNSIGNED DIVIDE 
      OUTPUT S                SIGNED DIVIDE 
   CASE_END 
   OUTPUT ".W"
   POSITION ABS,TAG_COLUMN      MOVE OVER TO TAG FIELD
   CALL EA_DISP                 DISPLAY THE EFFECTIVE ADDRESS 
   OUTPUT COMMA 
   GOTO DATA_REG_11_9           DISPLAY THE DATA REGISTER IN BITS 11-9
  
SBCD_ABCD 
   CASE_OF 14,14
      OUTPUT S
      OUTPUT A
   CASE_END 
   OUTPUT "BCD.B" 
   POSITION ABS,TAG_COLUMN      MOVE OVER TO THE OPERAND FIELD
   IF 3,3 = 0 THEN GOTO REG_TO_REG
      OUTPUT "-[" 
      CALL ADDR_REG_2_0         DISPLAY THE SOURCE ADDR REGISTER
      OUTPUT "],-[" 
      CALL ADDR_REG_11_9        DISPLAY THE DESTINATION ADDR REGISTER 
      OUTPUT RIGHT_BRACKET
      RETURN
  
REG_TO_REG
   CALL DATA_REG_2_0            DISPLAY THE SOURCE DATA REGISTER
   OUTPUT COMMA 
   GOTO DATA_REG_11_9           DISPLAY THE DESTINATION REGISTER
  
EXG 
   OUTPUT "EXG.L" 
   POSITION ABS,TAG_COLUMN
   IF 7,3 = 01000B THEN SET TEMP1,0              DATA REGISTER EXG
   IF 7,3 = 01001B THEN SET TEMP1,1              ADDR REGISTER EXG
   IF 7,3 = 10001B THEN SET TEMP1,2              ADDR/DATA REGISTER EXG 
   CASE_OF TEMP1
      CALL DATA_REG_11_9        DISPLAY THE FIRST DATA REGISTER 
      CALL ADDR_REG_11_9        DISPLAY THE FIRST ADDR REGISTER 
      CALL DATA_REG_11_9        DISPLAY THE DATA REGISTER 
      NOP                       CANT GET HERE 
   CASE_END 
   OUTPUT COMMA 
   LOAD INITIAL_DATA            GET THE ORIGINAL OPCODE BACK
   CASE_OF TEMP1
      CALL DATA_REG_2_0         DISPLAY THE SECOND DATA REGISTER
      CALL ADDR_REG_2_0         DISPLAY THE SECOND ADDR REGISTER
      CALL ADDR_REG_2_0         DISPLAY THE ADDRESS REGISTER
      NOP                       CANT GET HERE 
   CASE_END 
   RETURN 
  
  
************************************************************************
  
Shift_Rotate
   IF 7,6 = 11B THEN GOTO Memory_Shift
   CASE_OF 4,3                  DISPLAY THE TYPE OF SHIFT 
      OUTPUT "AS" 
      OUTPUT "LS" 
      OUTPUT "ROX"
      OUTPUT "RO" 
   CASE_END 
   CASE_OF 8,8                  DISPLAY THE DIRECTION OF THE SHIFT
      OUTPUT R
      OUTPUT L
   CASE_END 
   CALL SHOW_SIZE               DISPLAY THE OPERATION SIZE (B,L,W)
  
   IF 5,5 = 1 THEN GOTO REG_COUNT 
      CALL IMM_DATA_11_9        DISPLAY THE IMMEDIATE DATA
      GOTO COUNT_DISPLAYED
  
REG_COUNT 
   CALL DATA_REG_11_9           DISPLAY THE DATA REG IN BITS 11-9 
  
COUNT_DISPLAYED 
   OUTPUT COMMA 
   GOTO DATA_REG_2_0            DISPLAY THE SELECTED DATA REGISTER
  
************************************************************************
  
Memory_Shift
   CASE_OF 10,9                 DISPLAY THE TYPE OF SHIFT 
      OUTPUT "AS" 
      OUTPUT "LS" 
      OUTPUT "ROX"
      OUTPUT "RO" 
   CASE_END 
   CASE_OF 8,8                  DISPLAY THE DIRECTION OF THE SHIFT
      OUTPUT R
      OUTPUT L
   CASE_END 
   OUTPUT ".W"                  WORD OPERATIONS ONLY
   SET SIZE_FIELD,WORD          SET OPERATION SIZE TO WORD
   POSITION ABS,TAG_COLUMN      MOVE OVER TO THE TAG DISPLAY FIELD
   GOTO EA_DISP                 DISPLAY THE EFFECTIVE ADDRESS 
  
************************************************************************
  
UNIMPLEMENTED 
   OUTPUT "Unimplemented Instruction:"
   OUTPUT ACCUMULATOR,HEX4_FMT
   SET PREFETCH_LOC,+1          RELATIVE POSITION OF EXPECTED PREFETCH
   GOTO MARK_PREFETCH           DUMP THE UNUSED PREFETCH
  
************************************************************************
  
*  68000 INVERSE ASSEMBLER UTILITY ROUTINES 
  
************************************************************************
  
EA_DISP 
  
*  DISPLAY AN EFFECTIVE ADDRESS 
  
SOURCE_EA 
   SET EA_TYPE,1                INDICATE A "NORMAL" LOCATION EA 
   GOTO EA_TYPE_SET 
DESTINATION_EA
   SET EA_TYPE,0                INDICATE A "SPECIAL" LOCATION EA
  
EA_TYPE_SET 
   LOAD INITIAL_DATA            GET THE ORIGINAL OPCODE 
   CASE_OF EA_TYPE
      ROTATE RIGHT,6            MOVE MODE FIELD TO LSB (SPECIAL)
      ROTATE RIGHT,3            MOVE MODE FIELD TO LSB (NORMAL) 
   CASE_END 
   AND ML3                      GET THE  LOWER THREE BITS 
   STORE MODE_FIELD             SAVE THE MODE FIELD 
  
  
   LOAD INITIAL_DATA            GET THE ORIGINALLY INPUT OPCODE 
   CASE_OF EA_TYPE
      ROTATE RIGHT,9            MOVE REGISTER FIELD TO LSB (SPECIAL)
      NOP                       NO MOVE REQUIRED (NORMAL) 
   CASE_END 
  
EA_DISP_MODE_SET
  
*  ENTER HERE IF THE MODE HAS BEEN PRESET.  THIS ASSUMES THE REGISTER 
*  NUMBER TO BE IN THE LOWER 3 BITS.
  
   AND ML3                      GET THE LOWER 3 BITS
   STORE REG_FIELD              SAVE THE REGISTER FIELD 
  
   CASE_OF MODE_FIELD 
      OUTPUT D                DATA REGISTER DIRECT
      OUTPUT A                ADDR REGISTER DIRECT
      OUTPUT "[A"               ADDR REGISTER INDIRECT
      OUTPUT "[A"               ADDR REGISTER INDIRECT W/POSTINCREMENT
      OUTPUT "-[A"              ADDR REGISTER INDIRECT W/PREDECREMENT 
      GOTO DISP16               ADDR REGISTER INDIRECT W/DISPLACEMENT 
      GOTO DISP8_INDEX          ADDR REG INDIRECT W/DISPLACEMENT&INDEX
      GOTO SPECIAL              SPECIAL ADDRESSING MODES
   CASE_END 
  
   LOAD REG_FIELD               GET THE REGISTER FIELD
   OUTPUT ACCUMULATOR,DEC1_FMT_3  DISPLAY THE SELECTED REGISTER NUMBER
  
   LOAD INITIAL_DATA            GET THE ORIGINAL OPCODE BACK
   CASE_OF MODE_FIELD 
      RETURN                    DATA REGISTER DIRECT
      RETURN                    ADDRESS REGISTER DIRECT 
      OUTPUT RIGHT_BRACKET                ADDRESS REGISTER INDIRECT 
      OUTPUT "]+"               ADDR REGISTER INDIRECT W/POSTINCREMENT
      OUTPUT RIGHT_BRACKET                ADDR REGISTER INDIRECT W/PREDECREMENT 
      NOP                       SHOULDNT EVER GET HERE
      NOP                       SHOULDNT EVER GET HERE
      NOP                       SHOULDNT EVER GET HERE
   CASE_END 
   RETURN 
  
************************************************************************
  
DISP16
  
*   16 BIT DISPLACEMENT MODE
  
   CALL DISP_NEXT_WORD          DISPLAY THE 16 BIT OFFSET VALUE 
   OUTPUT "[A"
   LOAD REG_FIELD               GET THE REGISTER INDICATED
   OUTPUT ACCUMULATOR,DEC1_FMT_3  DISPLAY THE SELECTED REGISTER 
   OUTPUT RIGHT_BRACKET 
   LOAD INITIAL_DATA            GET THE ORIGINAL OPCODE BACK
   RETURN 
  
************************************************************************
  
DISP8_INDEX 
  
*  8 BIT DISPLACEMENT AND INDEX MODE
  
   CALL READ_NEXT_OPERAND         GET THE EXTENSION WORD
  
   IF INPUT_ERROR = 0 THEN GOTO DISP8_OK
      OUTPUT "**[A"             ADDRESS REG WILL BE BACKFILLED
      LOAD REG_FIELD            GET THE ADDRESS REGISTER NUMBER 
      OUTPUT ACCUMULATOR,DEC1_FMT_3               DISPLAY REG # 
      OUTPUT ",**.*]" 
      RETURN
  
DISP8_OK
   AND ML8                      GET THE 8 BIT DISPLACEMENT
   OUTPUT ACCUMULATOR,HEX2_FMT  DISPLAY THE 8 BIT DISPLACEMENT
   OUTPUT "[" 
   LOAD INPUT_DATA              GET THE ORIGINAL INPUT DATA BACK
   OUTPUT A 
   LOAD REG_FIELD               GET THE ADDRESS REGISTER NUMBER 
   OUTPUT ACCUMULATOR,DEC1_FMT_3  DISPLAY THE ADDRESS REGISTER NUMBER 
   OUTPUT COMMA 
   CASE_OF 15,15
      OUTPUT D                DATA REGISTER IS INDEX
      OUTPUT A                ADDRESS REGISTER IS INDEX 
   CASE_END 
   LOAD INPUT_DATA              GET THE EXTENSION WORD
   ROTATE RIGHT,12              MOVE REGISTER NUMBER TO LSB 
   OUTPUT ACCUMULATOR,DEC1_FMT_3  DISPLAY THE REGISTER NUMBER 
   OUTPUT PERIOD
   LOAD INPUT_DATA              GET THE EXTENSION WORD
   CASE_OF 11,11
      OUTPUT W
      OUTPUT L
   CASE_END 
   OUTPUT RIGHT_BRACKET 
   LOAD INITIAL_DATA            RESTORE THE ORIGINAL OPCODE 
   RETURN 
  
************************************************************************
  
SPECIAL 
   CASE_OF REG_FIELD
      GOTO ABS_SHORT            ABSOLUTE SHORT ADDRESS
      GOTO ABS_LONG             ABSOLUTE LONG ADDRESS 
      GOTO PC_DISPLACEMENT      PC WITH DISPLACEMENT
      GOTO PC_INDEX             PC WITH DISPLACEMENT & INDEX
      GOTO IMMEDIATE_SR         IMMEDIATE/ SR CCR MODIFY
   CASE_END 
   GOTO ILLEGAL_OPERAND         ILLEGAL ADDRESSING MODES
  
************************************************************************
  
ABS_SHORT 
   CALL READ_NEXT_OPERAND       GET THE EXTENSION WORD
   IF INPUT_ERROR = 0 THEN GOTO ABS_SHORT_OK
      OUTPUT "******" 
      RETURN
ABS_SHORT_OK
   IF 15,15 = 1 THEN INCLUSIVE_OR 0FF0000H
   GOTO ADDRESS_OUTPUT
  
************************************************************************
  
ABS_LONG
   CALL READ_NEXT_OPERAND       GET THE HIGH ORDER BYTE 
   STORE HIGH_BYTE              ... AND SAVE IT 
   SET RD_STATUS,00B            INITIAL STATUS...NO ERRORS
   IF INPUT_ERROR <> 0 THEN SET RD_STATUS,01B   HIGH BYTE FAILURE 
   CALL READ_NEXT_OPERAND       GET THE LOW ORDER BYTE
   STORE LOW_BYTE               ... AND SAVE IT 
   LOAD RD_STATUS               GET THE CURRENT READ STATUS 
   IF INPUT_ERROR <> 0 THEN INCLUSIVE_OR 10B    LOW BYTE FAILURE
   STORE RD_STATUS              SAVE THE READ STATUS
  
   IF RD_STATUS = 00B THEN GOTO MAP_32          BRIF MAPPING IS OK
      LOAD HIGH_BYTE            GET THE UPPER 16 BIT BYTE 
      CASE_OF RD_STATUS 
         NOP                    CANT GET HERE 
         OUTPUT FOUR_STAR                       01 = HIGH BYTE FAILURE
         OUTPUT ACCUMULATOR,HEX4_FMT            10 = HIGH BYTE OK 
         OUTPUT FOUR_STAR                       11 = BOTH BYTE FAILURE
      CASE_END
      LOAD LOW_BYTE             GET THE LOWER 16 BIT BYTE 
      CASE_OF RD_STATUS 
         NOP                    CANT GET HERE 
         OUTPUT ACCUMULATOR,HEX4_FMT            01 = LOW BYTE OK
         OUTPUT FOUR_STAR                       10 = LOW BYTE FAILURE 
         OUTPUT FOUR_STAR                       11 = BOTH BYTE FAILURE
      CASE_END
      RETURN
  
MAP_32
   LOAD HIGH_BYTE               GET THE UPPER 16 BITS 
   AND 0FFH                     MASK TO LOWER 2 HIGH BYTE DIGITS
   ROTATE LEFT,16               MOVE TO THE UPPER 8 BITS
   INCLUSIVE_OR LOW_BYTE        OR IN THE LOWER 16 BITS 
   GOTO ADDRESS_OUTPUT
  
************************************************************************
  
PC_DISPLACEMENT 
   CALL READ_NEXT_OPERAND          GET THE 16 BIT DISPLACEMENT
   IF INPUT_ERROR <> 0 THEN GOTO WORD_ERROR 
PC_DISP_OK
   IF 15,15 = 1 THEN INCLUSIVE_OR 0FF0000H    SIGN EXTEND DISPLACEMENT
   ADD INPUT_ADDRESS            ADD DISPLACEMENT TO THE CURRENT PC
   CALL ADDRESS_OUTPUT
   OUTPUT "[PC]"
   RETURN 
  
************************************************************************
  
PC_INDEX
   CALL READ_NEXT_OPERAND       GET THE EXTENSION WORD
   IF INPUT_ERROR = 0 THEN GOTO PC_INDEX_OK 
      OUTPUT "**[PC,**.*]"
      RETURN
PC_INDEX_OK 
   AND 0FFH                                   IGNORE UPPER BYTE 
   IF 7,7 = 1 THEN INCLUSIVE_OR 0FFFF00H    SIGN EXTEND DISPLACEMENT
   ADD INPUT_ADDRESS            ADD DISPLACEMENT TO THE CURRENT PC
   CALL ADDRESS_OUTPUT
   LOAD INPUT_DATA
   OUTPUT "[PC,"
   CASE_OF 15,15
      OUTPUT D
      OUTPUT A
   CASE_END 
   ROTATE RIGHT,12              MOVE INDEX REGISTER TO LSB
   OUTPUT ACCUMULATOR,DEC1_FMT_3                DISPLAY REG NUMBER
   OUTPUT PERIOD
   LOAD INPUT_DATA              GET THE EXTENSION WORD BACK 
   CASE_OF 11,11                DISPLAY THE SIZE OF THE INDEX 
      OUTPUT W
      OUTPUT L
   CASE_END 
   OUTPUT RIGHT_BRACKET 
   RETURN 
  
************************************************************************
  
IMMEDIATE_SR
   IF SR_ACCESS = FALSE THEN GOTO IMMEDIATE 
      CASE_OF SIZE_FIELD
         OUTPUT CCR           BYTE TRANSFER TO CCR
         OUTPUT SR            WORD TRANSFER EFFECTES ENTIRE SR
         GOTO ILLEGAL_OPERAND 
         GOTO ILLEGAL_OPERAND 
      CASE_END
      RETURN
IMMEDIATE 
   OUTPUT NUM_SIGN
   CALL READ_NEXT_OPERAND          GET THE NEXT OPERAND WORD
   IF INPUT_ERROR = 0 THEN GOTO EXT1_OK 
      CASE_OF SIZE_FIELD
         OUTPUT TWO_STAR        NO BYTE FOUND 
         OUTPUT FOUR_STAR       NO WORD FOUND 
         OUTPUT FOUR_STAR       HIGH ORDER WORD NOT FOUND 
         GOTO ILLEGAL_OPCODE    ILLEGAL SIZE SPECIFICATION
      CASE_END
      GOTO CHECK_FOR_LONG 
EXT1_OK 
   CASE_OF SIZE_FIELD 
      OUTPUT ACCUMULATOR,HEX2_FMT               DISPLAY 8 BIT DATA
      OUTPUT ACCUMULATOR,HEX4_FMT               DISPLAY 16 BIT DATA 
      OUTPUT ACCUMULATOR,HEX4_FMT               DISP HIGH ORDER 16 BITS 
      GOTO ILLEGAL_OPCODE                       SHOULDNT EVER GET HERE
   CASE_END 
  
CHECK_FOR_LONG
* 
*  NOW, SEE IF WE ARE DEALING WITH LONG IMMEDIATE VALUES.  IF NOT,
*  THERE IS NO NEED TO READ A SECOND EXTENSION WORD.
* 
   IF SIZE_FIELD <> 10B THEN RETURN             BRIF NOT LONG DATA
* 
*  AT THIS POINT, WE ARE LOOKING AT "LONG" IMMEDIATE VALUES ONLY
* 
   CALL READ_NEXT_OPERAND          GET THE 2ND EXTENSION WORD 
   IF INPUT_ERROR <> 0 THEN GOTO WORD_ERROR 
EXT2_OK 
   OUTPUT ACCUMULATOR,HEX4_FMT
   RETURN 
  
************************************************************************
  
DISP_NEXT_WORD
  
*  DISPLAY THE NEXT WORD (16 BIT QUANTITY) FROM THE ANALYZER MEMORY 
  
   CALL READ_NEXT_OPERAND         GET THE NEXT STATE
  
   IF INPUT_ERROR <> 0 THEN GOTO WORD_ERROR 
WORD_READ_OK
   OUTPUT ACCUMULATOR,HEX4_FMT
   RETURN 
  
************************************************************************
  
READ_NEXT_OPERAND 
  
   INCREMENT REQUESTED_ADDRESS
   INCREMENT REQUESTED_ADDRESS
   CALL EXMODE_PW 
  
   CALL CHECK_FOR_BLOCK         DONT ALLOW IT TO CROSS BLOCK BOUNDARIES 
  
   IF INPUT_ERROR = 0 THEN TAG_WITH OPERAND_USED     MARK THIS AS A USED STATE
  
SKIP_RET
   LOAD INPUT_DATA
   RETURN 
  
  
EXMODE_PW 

BCMODE_PW 
     INPUT ABS,REQUESTED_ADDRESS,QUALIFIED      READ THE NEXT STATE 
     RETURN 
  
READ_MEMORY_OPERAND 
  
************************************************************************
  
CHECK_FOR_BLOCK 
  
   IF TASK = 3 THEN RETURN     DISABLE FOR STATE (TEMPORARY)
  
*  THIS ROUTINE CHECKS TO BE SURE THAT AN INVERSE ASSEMBLY BLOCK WILL 
*  NOT BE CROSSED INTO BY AN INSTRUCTION BEING DECODED IN A PREVIOUS
*  BLOCK.  FOR MORE INFORMATION ON THIS PROCEDURE, SEE STEVE WILLIAMS.
* 
*  THE OPERATION OF THIS ROUTINE: 
* 
*      1.  IF A DISASSEMBLY BLOCK HAS BEEN PREVIOUSLY ENCOUNTERED 
*          DURING THE DECODING OF THE CURRENT INSTRUCTION, THIS 
*          ROUTINE ASSUMES THAT WE ARE STILL IN THAT BLOCK, AND 
*          SETS "INPUT_ERROR" TO MAKE THE INVERSE ASSEMBLER THINK 
*          THAT A READ ERROR DID OCCUR. 
* 
*      2.  IF THERE WAS A NORMAL ERROR IN READING THIS STATE, NO
*          DISASSEMBLY BLOCK IS ASSUMED, BUT "INPUT_ERROR" DOES 
*          REFLECT THAT AN ERROR OCCCURRED. 
* 
*      3.  IF THE CONTROL TAG (BITS 17 AND 16 OF INPUT_TAG) INDICATES 
*          THIS STATE TO HAVE BEEN PREVIOUSLY USED AS A FIRST INSTRUCTION,
*          THEN A DISASSEMBLY BLOCK BOUNDARY IS SAID TO HAVE
*          ENCOUNTERED.  THE DISASSEMBLY_BLOCK FLAG IS SET, AND 
*          AN INPUT ERROR IS FLAGGED SO THAT THE INVERSE ASSEMBLER
*          CAN DO THE PROPER THINGS WITH THE ERROR FLAG.
  
   IF DISASSEMBLY_BLOCK = TRUE THEN SET INPUT_ERROR,1 
  
   IF INPUT_ERROR <> 0 THEN RETURN              LEAVE IF READ ERROR 
  
   LOAD INPUT_TAG            GET THE TAG ASSOCIATED WITH THE NEW STATE
   IF 17,16 <> 11B THEN RETURN   11B CONTROL TAG IS ONLY TAG NOT OK TO USE. 
  
   SET DISASSEMBLY_BLOCK,TRUE             A DISASSEMBLY BLOCK IS PRESENT
   SET INPUT_ERROR,1      MAKE IT LOOK LIKE THE READ FAILED 
   RETURN 
  
************************************************************************
  
SHOW_SIZE 
  
*  DISPLAY THE SIZE OF THE OPERATION (BYTE, WORD, LONG) 
  
   OUTPUT PERIOD
   CASE_OF 7,6
      OUTPUT B
      OUTPUT W
      OUTPUT L
      GOTO ILLEGAL_OPERAND
   CASE_END 
   ROTATE RIGHT,6               MOVE SIZE TO LSB
   AND ML2                      ONLY LOWER 2 BITS ARE THE SIZE FIELD
   STORE SIZE_FIELD             SAVE THE SIZE FIELD 
   LOAD INITIAL_DATA            RESTORE THE ORIGINAL OPCODE 
   POSITION ABS,TAG_COLUMN      MOVE OVER TO OPERAND FIELD
   RETURN 
  
************************************************************************
  
IMM_DATA_11_9 
  
*  DISPLAY THE 3 BIT IMMEDIATE DATA 
*  VALUES 1-7 ARE INTERPRETED AS 1-7, VALUE 0 IS INTERPRETED AS 8 
  
   OUTPUT NUM_SIGN                IMMEDIATE DATA
   IF 11,9 <> 0 THEN GOTO IMM_NOT_0 
      OUTPUT "8"
      RETURN
IMM_NOT_0 
      ROTATE RIGHT,9            MOVE DATA TO LOWER 3 BITS 
      OUTPUT ACCUMULATOR,DEC1_FMT_3               DISPLAY THE 3 BIT # 
      LOAD INITIAL_DATA         RESTORE THE INITIAL OPCODE
      RETURN
  
************************************************************************
  
CONDITION_CODE
  
*  DISPLAY THE APPLICABLE CONDITION CODES FOR Scc, DBcc AND Bcc INSTR 
  
   CASE_OF 11,8 
      GOTO COND_T_RA            RA FOR Bcc, T FOR Scc AND DBcc
      GOTO COND_F_BSR           BSR, F FOR Scc AND DBcc 
      OUTPUT "HI" 
      OUTPUT "LS" 
      OUTPUT "CC" 
      OUTPUT "CS" 
      OUTPUT "NE" 
      OUTPUT "EQ" 
      OUTPUT "VC" 
      OUTPUT "VS" 
      OUTPUT "PL" 
      OUTPUT "MI" 
      OUTPUT "GE" 
      OUTPUT "LT" 
      OUTPUT "GT" 
      OUTPUT "LE" 
   CASE_END 
   RETURN 
  
COND_T_RA 
  
*  DISPLAY A "T" (TRUE) FOR Scc AND DBcc. "RA" FOR Bcc INSTR
  
   CASE_OF 12,12
      OUTPUT "RA"               BRA INSTRUCTION 
      OUTPUT "T"                ST AND DBT INSTRUCTIONS 
   CASE_END 
   RETURN 
  
COND_F_BSR
  
*  DISPLAY AN "F" (FALSE) FOR Scc AND DBcc, OR BSR
  
   CASE_OF 12,12
      OUTPUT SR               BSR INSTRUCTION 
      OUTPUT "F"                SF AND DBF INSTRUCTIONS 
   CASE_END 
   RETURN 
  
************************************************************************
  
ILLEGAL_OPERAND 
   POSITION ABS,TAG_COLUMN
   OUTPUT ILLEGAL 
   OUTPUT " Operand    "
   ABORT
  
************************************************************************
  
DATA_ERROR
   OUTPUT "Data Error"
   ABORT
  
************************************************************************
  
ILLEGAL_OPCODE
   OUTPUT ILLEGAL 
   OUTPUT " Opcode" 
   ABORT
  
************************************************************************
  
WORD_ERROR
    OUTPUT FOUR_STAR
    RETURN
  
************************************************************************
  
INCOMPLETE_OPCODE 
     OUTPUT "incomplete opcode" 
     ABORT
  
  
************************************************************************
  
ADDR_REG_11_9 
  
*  DISPLAY THE ADDRESS REGISTER FOUND IN BITS 11 THRU 9 
  
   LOAD INITIAL_DATA            BE SURE THE INITIAL OPCODE IS LOADED
   ROTATE RIGHT,9               MOVE THE REGISTER FIELD TO 3 LSB
  
ADDR_REG_2_0
   OUTPUT A 
   GOTO REG_TYPE_SHOWN
  
DATA_REG_11_9 
  
*  DISPLAY THE DATA REGISTER FOUND IN BITS 11 THRU 9
  
   LOAD INITIAL_DATA            BE SURE THE INITIAL OPCODE IS LOADED
   ROTATE RIGHT,9               MOVE THE REGISTER FIELD TO 3 LSB
  
DATA_REG_2_0
   OUTPUT D 
  
REG_TYPE_SHOWN
  
*  DISPLAY THE REGISTER NUMBER FOUND IN THE REG FIELD (BITS 11 THRU 9)
  
   OUTPUT ACCUMULATOR,DEC1_FMT_3  DISPLAY THE REGISTER NUMBER 
   LOAD INITIAL_DATA            RESTORE THE ORIGINAL OPCODE 
   RETURN 
  
************************************************************************
  
EXTENDED_MATH 
  
*  DISPLAY "X", SIZE AND OPERANDS FOR ADDX AND SUBX INSTRUCTIONS
  
   OUTPUT "X" 
   CALL SHOW_SIZE               DISPLAY THE OPERATION SIZE
   CASE_OF 3,3
      SET MODE_FIELD,000B       SRC & DEST ARE DATA REG DIRECT MODE 
      SET MODE_FIELD,100B       SRC & DEST ARE PREDEC ADDR IND MODE 
   CASE_END 
   CALL EA_DISP_MODE_SET        DISPLAY THE SOURCE OPERAND
   ROTATE RIGHT,9               MOVE DESTINATION TO LOWER 3 BITS
   OUTPUT COMMA 
   GOTO EA_DISP_MODE_SET       DISPLAY THE DESTINATION OPERAND
  
************************************************************************
  
MARK_PREFETCH 
  
* 
*  FOR INSTRUCTIONS THAT HAVE KNOWN, UNUSED PREFETCH STATES, FIND THE 
*  NEXT OPCODE STATE, THEN SEE IF ITS ADDRESS IS THE ADDRESS IMMEDIATELY
*  FOLLOWING THE CURRENT INSTRUCTION.  IF SO, THEN ITS UNUSED PREFETCH. 
* 
  
   INPUT REL,PREFETCH_LOC,QUALIFIED  LOOK FOR THE NEXT PROGRAM READ STATE 
   CALL CHECK_FOR_BLOCK         DONT CROSS INVERSE ASSEMBLY BLOCK 
   IF INPUT_ERROR <> 0 THEN RETURN   NO MORE PROG READ STATUS STATES
   LOAD REQUESTED_ADDRESS       ADDR OF STATE SUSPECTED TO HAVE UNUSED PREFETCH 
   ADD 2                        MOVE AHEAD TO POINT AT UNUSED PREFETCH
   SUBTRACT INPUT_ADDRESS       SUBTRACT OFF THE CURRENT ADDRESS
   IF 31,0 <> 0 THEN RETURN 
   TAG_WITH UNUSED_PREFETCH 
   RETURN 
  
************************************************************************
  
UNKNOWN 
  
* 
*  THIS IS TO DISPLAY THE FUNCTION CODES OF UNKNOWN MEMORY CYCLES 
* 
  
     OUTPUT "unknown (" 
     AND  01110000B 
     ROTATE RIGHT,4 
     OUTPUT ACCUMULATOR,BIN3_FMT
     OUTPUT ")" 
     LOAD INPUT_STATUS
     RETURN 
************************************************************************
INT_ACK 
  
* 
* THIS ROUTINE CHECKS FOR INT_ACK OR CPU SPACE OPERATION
* 
  
    LOAD INPUT_ADDRESS
    SET INT,TRUE
    IF 23,4 <> INT_VAL THEN GOTO CPU
    OUTPUT "int_ack"
    LOAD INPUT_STATUS 
    RETURN

CPU 
    SET INT,FALSE 
    OUTPUT "cpu space"
    LOAD INPUT_STATUS 
    RETURN
  
************************************************************************
